<?php
namespace Codexpert\CoDesigner_Pro\Modules;
use Codexpert\CoDesigner\Helper;
use Codexpert\Plugin\Base;
use Elementor\Plugin as Elementor;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit; 
}

class Invoice_Builder extends Base {

    public $slug;
    public $version;
    public $id = 'codesigner-invoice';

    /**
	 * Constructor
	 */
	public function __construct() {
        include_once( dirname( __FILE__ ) . '/inc/functions.php' );

		$this->plugin	= get_plugin_data( CODESIGNER );
		$this->slug		= $this->plugin['TextDomain'];
		$this->version	= $this->plugin['Version'];

        // create new folder for pdf invoices
        $invoice_dir    = codesigner_get_invoice_dir();
        if ( ! is_dir( $invoice_dir ) ) wp_mkdir_p( $invoice_dir );

        // register widget
		$this->action( 'elementor/widgets/register', 'register_widget' );

        // register scripts
		$this->action( 'wp_enqueue_scripts', 'enqueue_scripts' );
        $this->action( 'elementor/editor/after_enqueue_scripts', 'enqueue_scripts' );

        // reginster template type
        $this->filter( 'codesigner-pro_template_types', 'add_invoice_template_type' );

        // add invoice to mail
        $this->filter( 'woocommerce_email_attachments', 'add_invoice_pdf_to_email', 10, 3 );

        // show invoice download button in my account page
        $this->filter( 'woocommerce_my_account_my_orders_actions', 'display_invoice_download', 10, 2 );
    }

	public function __settings ( $settings ) {
		$invoice_templates	= Helper::get_posts( [ 'post_type' => 'elementor_library', 'meta_key' => '_elementor_template_type', 'meta_value' => 'wl-invoice' ] );
		$invoice_templates	= [ '' => __( 'Use WooCommerce Default', 'codesigner-pro' ) ] + $invoice_templates;
        $add_new_url        = add_query_arg( [ 'post_type' => 'elementor_library', 'tabs_group' => 'library', 'elementor_library_type' => 'wl-invoice' ], admin_url( 'edit.php' ) );
        $email_actions      = array(
            'new_order'                 => __( 'New Order', 'codesigner-pro' ),
            'customer_invoice'          => __( 'Customer Invoice', 'codesigner-pro' ),
            'failed_order'              => __( 'Failed Order', 'codesigner-pro' ),
            'cancelled_order'           => __( 'Cancelled Order', 'codesigner-pro' ),
            'customer_processing_order' => __( 'Processing Order', 'codesigner-pro' ),
            'customer_completed_order'  => __( 'Completed Order', 'codesigner-pro' ),
            'customer_refunded_order'   => __( 'Refunded Order', 'codesigner-pro' ),
            'customer_on_hold_order'    => __( 'On-hold Order', 'codesigner-pro' ),
        );

        $settings['sections'][ $this->id ] = [
            'id'        => $this->id,
            'label'     => __( 'Invoice Builder', 'codesigner-pro' ),
            'icon'      => 'dashicons-media-document',
            'sticky'	=> false,
            'fields'	=> [
                [
                    'id'      	=> 'invoice-template-id',
                    'label'     => __( 'Select Template', 'codesigner-pro' ),
                    'type'      => 'select',
                    'chosen'	=> true,
					'options' 	=> $invoice_templates,
                    'desc'      => sprintf( __( 'Select a template that\'ll used for <strong>%s</strong> generating invoice. <a href="%s" target="_blank">Create one</a> if you haven\'t created any.', 'codesigner-pro' ), __( 'Completed Order', 'codesigner-pro' ), $add_new_url ),

                ],
                [
                    'id'      	=> 'invoice-email-ids',
                    'label'     => __( 'Select Email Actions', 'codesigner-pro' ),
                    'type'      => 'select',
                    'chosen'	=> true,
                    'multiple'	=> true,
					'options' 	=> $email_actions,
					'default' 	=> ['customer_processing_order', 'customer_completed_order'],
                    'desc'   	=> __( 'Select the options when you want to send invoice to customers.', 'codesigner-pro' )
                ],
            ]
        ];

        return $settings;
    }

	public function register_widget( $widgets_manager ) {
		require_once( dirname( __FILE__ ) . '/widgets/invoice-builder-widget.php' );

		$widgets_manager->register( new \Invoice_Builder_Widget() );
	}

	public function enqueue_scripts() {
		wp_enqueue_script( "cd-invoice-builder-js", plugins_url( 'assets/js/admin.js', __FILE__), [ 'jquery' ], $this->version, true );
		wp_enqueue_style( "cd-invoice-builder-css", plugins_url( 'assets/css/invoice.css', __FILE__ ), '', $this->version, 'all' );
		wp_enqueue_media();
	}

    public function add_invoice_template_type( $types ) {
        $types['wl-invoice'] = [
            'class' => 'Invoice',
            'label' => __( 'CoDesigner Invoice', 'codesigner-pro' )
        ];

        return $types;
    }

    public function add_invoice_pdf_to_email( $attachments, $email_id, $order ) {
        global $cd_order_id;

        $cd_order_id            = $order->get_id();     // global variable to access in elementor widgets
        $order_id               = $order->get_id();
        $invoice_template_id    = codesigner_sanitize_number( Helper::get_option( 'codesigner-invoice', 'invoice-template-id' ) );
        $selected_email_ids     = Helper::get_option( 'codesigner-invoice', 'invoice-email-ids' );
        $template_content       = Elementor::$instance->frontend->get_builder_content( $invoice_template_id );
        $invoice_content        = codesigner_invoice_content_remove_styles( $template_content );
        $default_styles         = codesigner_get_default_invoice_css();
        $invoice_styles         = codesigner_get_invoice_template_css( $invoice_template_id );
        $invoice_pdf_html       = codesigner_get_invoice_template_html( $default_styles, $invoice_styles, $invoice_content );

        // validate template ID
        if ( ! $invoice_template_id ) return $attachments;

        /**
         * create invoice and return as attachment with email
         */
        if ( in_array( $email_id, $selected_email_ids ) ) {
            return $attachments[] = codesigner_generate_pdf_invoice( $order_id, $invoice_pdf_html );
        }

        return $attachments;
    }

    public function display_invoice_download( $actions, $order ) {
        $invoice_file_path      = codesigner_get_invoice_path( $order->get_id() );
        $invoice_download_url   = codesigner_get_invoice_download_url( $order->get_id() );

        if( ! file_exists( $invoice_file_path ) ) return $actions;

        $invoice_btn = array(
            'cd-invoice' => array(
                'url'  => $invoice_download_url,
                'name' => __( 'Invoice', 'codesigner-pro' ),
            )
        );

        return array_merge( $actions, $invoice_btn );
    }
}